<?php

namespace Database\Seeders;

use App\Models\User;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;

class DatabaseSeeder extends Seeder
{
    use WithoutModelEvents;

    /**
     * Seed the application's database.
     */
    public function run(): void
    {
        $roles = ['admin', 'provider', 'customer', 'support'];

        foreach ($roles as $role) {
            \Spatie\Permission\Models\Role::firstOrCreate(['name' => $role]);
        }

        $admin = User::firstOrCreate(
            ['email' => 'admin@example.com'],
            ['name' => 'Admin', 'password' => bcrypt('password')]
        );
        $admin->syncRoles(['admin']);

        $categories = [
            ['name' => 'Consulting', 'slug' => 'consulting'],
            ['name' => 'Design', 'slug' => 'design'],
            ['name' => 'Development', 'slug' => 'development'],
            ['name' => 'Home Services', 'slug' => 'home-services'],
        ];

        foreach ($categories as $category) {
            \App\Models\ServiceCategory::firstOrCreate(
                ['slug' => $category['slug']],
                ['name' => $category['name'], 'description' => $category['name']]
            );
        }

        $gateways = [
            ['name' => 'stripe', 'display_name' => 'Stripe', 'mode' => 'test', 'enabled' => true],
            ['name' => 'paypal', 'display_name' => 'PayPal', 'mode' => 'test', 'enabled' => false],
            ['name' => 'paystack', 'display_name' => 'Paystack', 'mode' => 'test', 'enabled' => false],
        ];

        foreach ($gateways as $gateway) {
            \App\Models\PaymentGateway::firstOrCreate(
                ['name' => $gateway['name']],
                $gateway
            );
        }

        $pages = [
            [
                'title' => 'Home',
                'slug' => 'home',
                'template' => 'home',
                'published' => true,
                'sections' => [
                    [
                        'type' => 'hero',
                        'title' => 'Book trusted services with escrow protection',
                        'subtitle' => 'Online and on-site providers with guaranteed payouts after completion.',
                        'cta_label' => 'Browse services',
                        'cta_url' => '/services',
                    ],
                    [
                        'type' => 'features',
                        'title' => 'Why customers choose us',
                        'items' => [
                            ['title' => 'Escrow-first', 'description' => 'Funds are held securely until you approve completion.'],
                            ['title' => 'Multi-gateway', 'description' => 'Stripe, PayPal, and Paystack coverage with instant webhooks.'],
                            ['title' => 'Calendar-aware', 'description' => 'Live availability with external busy-time blocking.'],
                        ],
                    ],
                    [
                        'type' => 'cta',
                        'title' => 'Become a provider',
                        'body' => 'List your service, control approvals, and receive payouts after jobs are done.',
                        'cta_label' => 'Create a service',
                        'cta_url' => '/provider',
                    ],
                ],
            ],
            [
                'title' => 'About',
                'slug' => 'about',
                'template' => 'about',
                'published' => true,
                'sections' => [
                    [
                        'type' => 'content',
                        'title' => 'Mission',
                        'body' => 'We connect customers with verified providers while enforcing fair, escrow-backed workflows.',
                    ],
                    [
                        'type' => 'stats',
                        'title' => 'Operations in numbers',
                        'items' => [
                            ['title' => '24h', 'description' => 'Default approval timeout for requests'],
                            ['title' => '48h', 'description' => 'Dispute window after service end'],
                            ['title' => '3+', 'description' => 'Payment rails supported at launch'],
                        ],
                    ],
                ],
            ],
            [
                'title' => 'FAQs',
                'slug' => 'faqs',
                'template' => 'faq',
                'published' => true,
                'sections' => [
                    [
                        'type' => 'faq',
                        'title' => 'Frequently asked questions',
                        'items' => [
                            ['title' => 'When is the provider paid?', 'description' => 'After completion and customer approval, or automatically after the grace period if no disputes are open.'],
                            ['title' => 'Can I reschedule?', 'description' => 'Yes, providers can propose new times and customers can accept or decline directly in the timeline.'],
                            ['title' => 'Which payments are supported?', 'description' => 'Stripe, PayPal, and Paystack with live webhook verification.'],
                        ],
                    ],
                ],
            ],
        ];

        foreach ($pages as $page) {
            \App\Models\Page::updateOrCreate(['slug' => $page['slug']], $page);
        }
    }
}
