<?php

namespace App\Filament\Resources;

use App\Filament\Resources\DisputeResource\Pages;
use App\Models\Dispute;
use Filament\Forms;
use Filament\Infolists;
use Filament\Resources\Resource;
use Filament\Schemas\Schema;
use Filament\Tables;
use Filament\Tables\Table;
use App\Filament\Resources\DisputeResource\RelationManagers\EventsRelationManager;

class DisputeResource extends Resource
{
    protected static ?string $model = Dispute::class;

    protected static \UnitEnum|string|null $navigationGroup = 'Operations';

    protected static \BackedEnum|string|null $navigationIcon = 'heroicon-o-exclamation-triangle';

    public static function form(Schema $schema): Schema
    {
        return $schema->schema([
            Forms\Components\Select::make('booking_id')->relationship('booking', 'id')->required(),
            Forms\Components\Select::make('opened_by')->relationship('opener', 'name')->required(),
            Forms\Components\Select::make('status')->options([
                'open' => 'open',
                'under_review' => 'under_review',
                'resolved' => 'resolved',
                'rejected' => 'rejected',
            ])->required(),
            Forms\Components\TextInput::make('reason')->required(),
            Forms\Components\Textarea::make('details'),
            Forms\Components\Textarea::make('resolution'),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('booking.id')->label('Booking'),
                Tables\Columns\TextColumn::make('opener.name')->label('Opened by'),
                Tables\Columns\TextColumn::make('status')->badge(),
                Tables\Columns\TextColumn::make('reason'),
                Tables\Columns\TextColumn::make('opened_at')->dateTime(),
                Tables\Columns\TextColumn::make('resolved_at')->dateTime(),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ]);
    }

    public static function infolist(Schema $schema): Schema
    {
        return $schema->schema([
            Infolists\Components\Section::make('Summary')
                ->schema([
                    Infolists\Components\TextEntry::make('status')->badge(),
                    Infolists\Components\TextEntry::make('reason'),
                    Infolists\Components\TextEntry::make('opened_at')->label('Opened')->dateTime(),
                    Infolists\Components\TextEntry::make('resolved_at')->label('Resolved')->dateTime(),
                    Infolists\Components\TextEntry::make('details')->columnSpanFull()->markdown(),
                    Infolists\Components\TextEntry::make('resolution')->columnSpanFull()->label('Resolution Notes')->markdown()->placeholder('Pending'),
                ])->columns(2),
            Infolists\Components\ViewEntry::make('timeline')
                ->label('Timeline & Evidence')
                ->view('filament.disputes.timeline')
                ->viewData([
                    'events' => fn (Dispute $record) => $record->events()->with('user')->latest()->get(),
                ]),
        ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListDisputes::route('/'),
            'create' => Pages\CreateDispute::route('/create'),
            'view' => Pages\ViewDispute::route('/{record}'),
            'edit' => Pages\EditDispute::route('/{record}/edit'),
        ];
    }

    public static function getRelations(): array
    {
        return [
            EventsRelationManager::class,
        ];
    }
}
